/*
 * Vale - a library for media streaming.
 *
 * tcp_test.c - A test suite for TCP abstraction handling.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tcp_tests.c,v 1.1.1.1 2007/04/03 13:13:37 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#if !defined(HAVE_GETADDRINFO) || !defined(HAVE_GETNAMEINFO)
#include <getaddrinfo.h>
#endif
#include <fcntl.h>

#include <event.h>

#include "vale.h"
#include "vale/unaligned.h"

typedef struct
{
    tcp_state_t *tcp;
    struct event evt;
    uint32_t tx_seq_no;
    uint32_t rx_seq_no;
    int tag;
} connection_info_t;

struct event timer_evt;
struct event monitor_evt;
connection_info_t group[1000];
int groups = 0;
uint8_t test_packet[256];
struct timeval target_time;

static void monitor_cb(int fd, short int event, void *arg)
{
}
/*- End of function --------------------------------------------------------*/

static void timeout_cb(int fd, short int event, void *arg)
{
    int i;

    for (i = 0;  i < groups;  i++)
    {
        put_unaligned_uint32(test_packet, htonl(i));
        put_unaligned_uint32(test_packet + sizeof(uint32_t), htonl(group[i].tx_seq_no++));
        if (tcp_socket_send(group[i].tcp, test_packet, 172, 0) < 0)
            printf("send error - %d\n", errno);
    }
}
/*- End of function --------------------------------------------------------*/

static void io_cb(int fd, short int event, void *arg)
{
    uint8_t buf[1000];
    int len;
    connection_info_t *s;

printf("DDD 10\n");
    s = (connection_info_t *) arg;
    len = tcp_socket_recv(s->tcp, buf, sizeof(buf), 0);
    if (len < 0  &&  errno != EAGAIN)
    {
printf("DDD 11\n");
        exit(2);
    }
    if (len >= 0)
    {
        //printf("%d - %5d\n", i, len);
        if (ntohl(get_unaligned_uint32(buf)) != s->tag)
            printf("%d: Bad channel %" PRIu32 " %d\n", s->tag, ntohl(get_unaligned_uint32(buf)), s->tag);
        if (ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))) != s->rx_seq_no)
        {
            if (s->rx_seq_no == 0)
                s->rx_seq_no = ntohl(get_unaligned_uint32(buf + sizeof(uint32_t)));
            else
                printf("%d: Bad sequence no %" PRIu32 " %" PRIu32 "\n", s->tag, ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))), s->rx_seq_no);
        }
        s->rx_seq_no++;
    }
printf("DDD 12\n");
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int i;
    struct sockaddr_in far;
    struct timeval tv;
    struct hostent *hp;
    const struct sockaddr_in *us;
    const char *far_host;
    int base_port;
    int far_base_port;
    int caller;
    int groups;

    far_host = "127.0.0.1";
    base_port = 10000;
    far_base_port = 12000;
    groups = 20;
    caller = FALSE;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-b") == 0)
        {
            base_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-c") == 0)
        {
            caller = TRUE;
            continue;
        }
        if (strcmp(argv[i], "-F") == 0)
        {
            far_base_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-f") == 0)
        {
            far_host = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-g") == 0)
        {
            groups = atoi(argv[++i]);
            continue;
        }
    }
 
    event_init();

    printf("Resolving far end server\n");
    if ((hp = gethostbyname(far_host)) == NULL) 
    {
        printf("Invalid address for far end server\n");
        exit(2);
    }
    far.sin_family = AF_INET;
    memcpy(&far.sin_addr, hp->h_addr, sizeof(far.sin_addr));
    far.sin_port = htons(far_base_port);
    
    printf("Initing TCP for %d groups\n", groups);
    evtimer_set(&monitor_evt, monitor_cb, &monitor_evt);
    evtimer_set(&timer_evt, timeout_cb, &timer_evt);
    for (i = 0;  i < groups;  i++)
    {
        memset(&group[i], 0, sizeof(group[i]));
        group[i].tcp = tcp_socket_create_and_bind(NULL, base_port, base_port + 2000);
        if (group[i].tcp == NULL)
        {
            printf("Failed at %d\n", i);
            exit(2);
        }
        /* The port randomisation on both sides should be similar, assuming none of the
           ports in range were already open. */
        us = tcp_socket_get_local(group[i].tcp);
        far.sin_port = htons(far_base_port + ntohs(us->sin_port) - base_port);
        tcp_socket_set_far(group[i].tcp, &far);
        event_set(&group[i].evt, tcp_socket_fd(group[i].tcp), EV_READ, io_cb, &group[i]);
        /* Add it to the active events, without a timeout */
        event_add(&group[i].evt, NULL);
        group[i].tag = i;
    }
#if 0
    if (!caller)
    {
        if (tcp_socket_listen(group[i].tcp) < 0)
        {
            printf("Listen failed - errno = %d", errno);
            exit(2);
        }
        /*endif*/
        FD_ZERO(&rfds);
        FD_SET(tcp_socket_fd(group[i].tcp), &rfds);
        if (select(tcp_socket_fd(group[i].tcp) + 1, &rfds, NULL, NULL, NULL) <= 0)
        {
            printf("Wait for accept failed - errno = %d", errno);
            exit(2);
        }
        while ((group[i].tcp = tcp_socket_accept(group[i].tcp)) == NULL)
        {
            if (errno != EAGAIN  &&  errno != EWOULDBLOCK)
            {
                printf("Accept failed - errno = %d", errno);
                exit(2);
            }
            /*endif*/
        }
        /*endwhile*/
    }
    else
    {
        FD_ZERO(&rfds_ref);
        FD_ZERO(&wfds_ref);
        max_fd = -1;
        for (i = 0;  i < groups;  i++)
        {
            if (tcp_socket_fd(group[i].tcp) > max_fd)
                max_fd = tcp_socket_fd(group[i].tcp);
            FD_SET(tcp_socket_fd(group[i].tcp), &wfds_ref);
        }
        if (tcp_socket_connect(group[i].tcp) < 0)
        {
            printf("Connect failed - errno = %d", errno);
        	exit (2);
        }
        printf("Connecting\n");

        tv.tv_sec = 0;
        tv.tv_usec = 0;
        for (;;)
        {
            memcpy(&rfds, &rfds_ref, sizeof(rfds));
            memcpy(&wfds, &wfds_ref, sizeof(wfds));
            tv.tv_sec = 0;
            tv.tv_usec = 20000 - tv.tv_usec;
            if (select(tcp_socket_fd(group[i].tcp) + 1, &rfds, &wfds, NULL, &tv) < 0)
            {
                printf("Wait for connect failed - errno = %d", errno);
                exit(2);
            }
            for (i = 0;  i < groups;  i++)
            {
                if (FD_ISSET(tcp_socket_fd(group[i].tcp), &rfds))
                {
                    len = tcp_socket_recv(group[i].tcp, buf, sizeof(buf), 0);
                    if (len < 0  &&  errno != EAGAIN)
                        exit(2);
#if 0
                    if (len >= 0)
                    {
                        //printf("%d - %5d\n", i, len);
                        if (ntohl(get_unaligned_uint32(buf)) != i)
                            printf("%d: Bad channel %" PRIu32 " %d\n", i, ntohl(get_unaligned_uint32(buf)), i);
                        if (ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))) != group[i].rx_seq_no)
                        {
                            if (group[i].rx_seq_no == 0)
                                group[i].rx_seq_no = ntohl(get_unaligned_uint32(buf + sizeof(uint32_t)));
                            else
                                printf("%d: Bad sequence no %" PRIu32 " %" PRIu32 "\n", i, ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))), group[i].rx_seq_no);
                        }
                        group[i].rx_seq_no++;
                    }
#endif
                }
                else if (FD_ISSET(tcp_socket_fd(group[i].tcp), &wfds))
                {
                    /* Writable means the connect has completed */
                    FD_CLR(tcp_socket_fd(group[i].tcp), &wfds_ref);
                    if (tcp_socket_connection_result(group[i].tcp) < 0)
                    {
                        printf("%d: Connection result %d\n", i, errno);
                    }
                    else
                    {
                        FD_SET(tcp_socket_fd(group[i].tcp), &rfds_ref);
                        printf("%d: Connected\n", i);
                    }
                }
            }
        }
    }
#endif
    timerclear(&tv);
    tv.tv_usec = 20000;
    gettimeofday(&target_time, NULL);
    timeradd(&target_time, &tv, &target_time);
    event_add(&timer_evt, &tv);

    timerclear(&tv);
    tv.tv_sec = 2;
    event_add(&monitor_evt, &tv);

    event_dispatch();

    printf("Done\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
