/*
 * Vale - a library for media streaming.
 *
 * queue_tests.c - Packet queue tests
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: queue_tests.c,v 1.5 2007/04/08 09:37:52 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>

#include <spandsp.h>
#include <spandsp-sim.h>

#include "vale.h"

#define STREAM_LEN  10000000

#define FALSE 0
#define TRUE (!FALSE)

rfc3550_state_t rfc3550_state;
rfc2198_state_t rfc2198_state;

static uint8_t buffer[STREAM_LEN];
static int total_rx_len;
static int next_seq_no = 0;
static uint32_t next_timestamp = 16000;

g1050_state_t *path;
frame_queue_state_t *queue;

uint64_t start;
uint64_t end;

/* Use a local random generator, so the results are consistent across platforms */
static int my_rand(void)
{
    static int rndnum = 1234567;

    return (rndnum = 1664525U*rndnum + 1013904223U);
}
/*- End of function --------------------------------------------------------*/

static int process_rx_packet(void *user_data,
                             const uint8_t buf[],
                             int len,
                             uint16_t seq_no,
                             uint32_t timestamp,
                             uint32_t ssrc,
                             int payload_type,
                             int mark)
{
    frame_queue_stats_t stats;

    printf("Got %d (%d)\n", seq_no, len);
frame_queue_dump(queue);
    if (seq_no != (next_seq_no & 0xFFFF))
    {
        printf("Test failed - seq_no %d, but expected %d (%d)\n", seq_no, next_seq_no, len);
frame_queue_stats(queue, &stats);
printf("%d %d %d %d %d %d %d\n", stats.frames_in, stats.frames_out, stats.frames_oos, stats.frames_repeats, stats.frames_late, stats.frames_missing, stats.frames_trimmed);
        exit(2);
    }
    if (timestamp != next_timestamp)
    {
        printf("Test failed - timestamp=%" PRIu32 ", next_timestamp=%" PRIu32 "\n", timestamp, next_timestamp);
        exit(2);
    }
    //printf("Seq %d, type %d, len %d\n", seq_no, payload_type, len);
    if (memcmp(buffer + total_rx_len, buf, len) != 0)
    {
        printf("Test failed - len=%d, total_rx_len=%d, %02x %02x %02x %02x\n", len, total_rx_len, buf[0], buffer[total_rx_len], buf[len - 1], buffer[total_rx_len + len - 1]);
        exit(2);
    }
    total_rx_len += len;
    next_seq_no++;
    next_timestamp += len;
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void back_to_back_test(void)
{
    int i;
    int msg_len;
    uint8_t buf[4096];
    uint8_t msg[4096];
    int len;
    int payload_type;
    int pos;
    int payload_ok[128];
    int sent_pkts;
    int dropped_pkts;
    //queue_state_t *queue;
    frame_queue_frame_t *frame;
    frame_queue_stats_t stats;
    int tx_seq_no;
    int rx_seq_no;
    timestamp_t tx_sender_stamp;
    timestamp_t rx_sender_stamp;
    double when;
    double tx_when;
    double rx_when;
    int seq_no;
    int rx_len;
    
    printf("Back to back lossy testing\n");
    if ((queue = frame_queue_new(256)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    if ((rfc3550_init(&rfc3550_state, 1)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    if ((rfc2198_init(&rfc2198_state, &rfc3550_state, 3, 42)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    tx_seq_no = 0;
    rx_seq_no = 0;
    tx_sender_stamp = 0;
    rx_sender_stamp = -1000;
    for (i = 0;  i <= 0x7F;  i++)
        payload_ok[i] = rfc3550_set_payload_handler(&rfc3550_state, i, frame_queue_put_rx_packet, queue);

    /* Generate some test data */
    for (i = 0;  i < STREAM_LEN;  i++)
        buffer[i] = my_rand() & 0xFF;

    /* Now pass it, in random sized pieces, through the protocol software */
    total_rx_len = 0;
    rx_len = 0;
    sent_pkts = 0;
    dropped_pkts = 0;
    when = 0.0;
    for (i = 0, pos = 0, payload_type = 0, tx_sender_stamp = 16000;  pos < STREAM_LEN - 256;  i++, pos += msg_len, payload_type++, tx_sender_stamp += msg_len)
    {
        /* Skip invalid payload types */
        while (payload_ok[payload_type & 0x7F])
            payload_type++;
        //printf("Pkt %d\n", i);
        msg_len = my_rand() & 0xFF;
        len = rfc2198_build_packet(&rfc2198_state, buffer + pos, msg_len, buf, tx_sender_stamp, payload_type & 0x7F, 0);
        /* Drop some packets. There is no protection here against dropping a burst
           long enough to cause the test to fail. */
        if (g1050_put(path, buf, len, i, tx_sender_stamp/8000.0) > 0)
        {
            //printf("Sending %d - %d %d\n", i, len, msg_len);
            sent_pkts++;
        }
        else
        {
            //printf("Dropping %d - %d %d\n", i, len, msg_len);
            dropped_pkts++;
        }
        while ((rx_len = g1050_get(path, msg, 1024, when, &seq_no, &tx_when, &rx_when)) >= 0)
        {
            //printf("Receiving %d - %d\n", seq_no, rx_len);
            rfc2198_process_rx_packet((void *) &rfc2198_state,
                                      msg,
                                      rx_len,
                                      seq_no,
                                      (int) (tx_when*8000.0 + 0.5),
                                      1234,
                                      42,
                                      0);
        }
        if ((frame = frame_queue_get_frame(queue, rx_sender_stamp)))
        {
            //printf("Got frame %d\n", frame->seq_no);
            process_rx_packet(NULL,
                              frame->data,
                              frame->len,
                              frame->seq_no,
                              frame->sender_stamp,
                              0,
                              frame->payload_type,
                              0);
            frame_queue_put_free_frame(queue, frame);
        }
        rx_sender_stamp += msg_len;
        when += msg_len/8000.0;
    }
    printf("Before final flushing: Queued frames %d, free frames %d\n", frame_queue_queued_frames(queue), frame_queue_free_frames(queue));
    /* Clear out the queue */
    rx_sender_stamp += 16000;
    when += 2.0;
    while ((rx_len = g1050_get(path, msg, 1024, when, &seq_no, &tx_when, &rx_when)) >= 0)
    {
        //printf("Sending %d - %d\n", i, len);
        rfc2198_process_rx_packet((void *) &rfc2198_state,
                                  msg,
                                  rx_len,
                                  seq_no,
                                  (int) (tx_when*8000.0 + 0.5),
                                  1234,
                                  42,
                                  0);
    }
    while ((frame = frame_queue_get_frame(queue, rx_sender_stamp)))
    {
        //printf("Got frame %d\n", frame->seq_no);
        process_rx_packet(NULL,
                          frame->data,
                          frame->len,
                          frame->seq_no,
                          frame->sender_stamp,
                          0,
                          frame->payload_type,
                          0);
        frame_queue_put_free_frame(queue, frame);
    }
    frame_queue_stats(queue, &stats);
    printf("%d %d %d %d %d %d %d\n", stats.frames_in, stats.frames_out, stats.frames_oos, stats.frames_repeats, stats.frames_late, stats.frames_missing, stats.frames_trimmed);

    printf("After final flushing: Queued frames %d, free frames %d\n", frame_queue_queued_frames(queue), frame_queue_free_frames(queue));
    printf("%d packets - %d sent, %d dropped, %d received\n", i, sent_pkts, dropped_pkts, next_seq_no);
    if (pos != total_rx_len)
    {
        printf("Test failed - %d %d\n", pos, total_rx_len);
        exit(2);
    }
}
/*- End of function --------------------------------------------------------*/

static void queue_only_test(void)
{
    int i;
    int j;
    frame_queue_state_t *queue;
    frame_queue_frame_t *frame;
    int tx_seq_no;
    int rx_seq_no;
    timestamp_t tx_sender_stamp;
    timestamp_t rx_sender_stamp;
    uint8_t data[160];

    if ((queue = frame_queue_new(160)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    tx_seq_no = 0;
    rx_seq_no = 0;
    tx_sender_stamp = 0;
    rx_sender_stamp = 0;

    printf("%d free frames\n", frame_queue_free_frames(queue));
    start = rdtscll();
    for (j = 0;  j < 100;  j++)
    {
        if ((frame = frame_queue_get_free_frame(queue)) == NULL)
        {
            printf("Could not get a free frame\n");
            exit(2);
        }
        for (i = 0;  i < 160;  i++)
            data[i] = i;
        /* Put these in the queue backwards, so they are badly
           out of order. */
        frame_queue_fill_frame(frame,
                               data,
                               160,
                               0,
                               99 - tx_seq_no,
                               160,
                               99*160 - tx_sender_stamp,
                               0);
        if (frame_queue_put_frame(queue, frame) < 0)
            frame_queue_put_free_frame(queue, frame);
        tx_seq_no++;
        tx_sender_stamp += 160;
    }
    printf("%d free frames\n", frame_queue_free_frames(queue));
    for (  ;  j < 1000000;  j++)
    {
        if ((frame = frame_queue_get_free_frame(queue)) == NULL)
        {
            printf("Could not get a free frame\n");
            exit(2);
        }
        for (i = 0;  i < 160;  i++)
            data[i] = i;
        frame_queue_fill_frame(frame,
                               data,
                               160,
                               0,
                               tx_seq_no,
                               160,
                               tx_sender_stamp,
                               0);
        if (frame_queue_put_frame(queue, frame) < 0)
            frame_queue_put_free_frame(queue, frame);
        tx_sender_stamp += 160;
        tx_seq_no++;
        rx_sender_stamp += 160;
        if ((frame = frame_queue_get_frame(queue, rx_sender_stamp)) == NULL)
        {
            printf("Could not get a frame at %d\n", j);
            exit(2);
        }
        if (frame->seq_no != (rx_seq_no & 0xFFFF))
        {
            printf("Seq no error at %d - %d %d\n", j, frame->seq_no, rx_seq_no);
            exit(2);
        }
        for (i = 0;  i < 160;  i++)
        {
            if (frame->data[i] != i)
            {
                printf("Data error at %d/%d\n", i, j);
                exit(2);
            }
        }
        rx_seq_no++;
        frame_queue_put_free_frame(queue, frame);
    }
    printf("%d free frames\n", frame_queue_free_frames(queue));
    for (  ;  j < 1000100;  j++)
    {
        rx_sender_stamp += 160;
        if ((frame = frame_queue_get_frame(queue, rx_sender_stamp)) == NULL)
        {
            printf("Could not get a frame at %d\n", j);
            exit(2);
        }
        if (frame->seq_no != (rx_seq_no & 0xFFFF))
        {
            printf("Seq no error at %d - %d %d\n", j, frame->seq_no, rx_seq_no);
            exit(2);
        }
        for (i = 0;  i < 160;  i++)
        {
            if (frame->data[i] != i)
            {
                printf("Data error at %d/%d\n", i, j);
                exit(2);
            }
        }
        rx_seq_no++;
        frame_queue_put_free_frame(queue, frame);
    }
    end = rdtscll();
    printf("%d free frames\n", frame_queue_free_frames(queue));

    rx_sender_stamp += 160;
    if ((frame = frame_queue_get_frame(queue, rx_sender_stamp)) != NULL)
    {
        printf("Could get a frame at %d\n", j);
        exit(2);
    }
    printf("Tests took %llu\n", end - start);
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int i;
    int model_no;
    int speed_pattern_no;

    model_no = 0;
    speed_pattern_no = 1;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-m") == 0)
        {
            model_no = argv[++i][0] - 'A' + 1;
            continue;
        }
        if (strcmp(argv[i], "-s") == 0)
        {
            speed_pattern_no = atoi(argv[++i]);
            continue;
        }
    }

    queue_only_test();
    
    srand48(0x1234567);
    if ((path = g1050_init(model_no, speed_pattern_no, 100, 50)) == NULL)
    {
        fprintf(stderr, "Failed to start IP network path model\n");
        exit(2);
    }
    g1050_dump_parms(model_no, speed_pattern_no);
    back_to_back_test();

    printf("Tests passed\n");
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
