/*
 * Vale - a library for media streaming.
 *
 * rfc3550.h - RTP core
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc3550.h,v 1.2 2007/04/07 03:37:25 steveu Exp $
 */

/*! \file */

#if !defined(_VALE_RFC3550_H_)
#define _VALE_RFC3550_H_

#define RFC3550_MAX_HEADER  64

typedef uint32_t rfc3550_timestamp_t;

typedef int (rtp_rx_packet_handler_t)(void *user_data,
                                      const uint8_t msg[],
                                      int len,
                                      uint16_t seq_no,
                                      rfc3550_timestamp_t ts,
                                      uint32_t ssrc,
                                      int payload_type,
                                      int mark);

#define RFC3550_RTCP_MAX_SDES 255      /* maximum text length for SDES */

typedef enum
{
    RTCP_SR    = 200,   /* 0x80 + 72 RFC3551 Sender report */
    RTCP_RR    = 201,   /* 0x80 + 73 RFC3551 Receiver report */
    RTCP_SDES  = 202,   /* 0x80 + 74 RFC3551 Source description */
    RTCP_BYE   = 203,   /* 0x80 + 75 RFC3551 Goodbye */
    RTCP_APP   = 204,   /* 0x80 + 76 RFC3551 Application defined */
    RTCP_RTPFB = 205,   /* 0x80 + 77 RFC4585 Generic RTP feedback */
    RTCP_PSFB  = 206,   /* 0x80 + 78 RFC4585 Payload feedback */
    RTCP_XR    = 207    /* 0x80 + 79 RFC3611 Extended reports */
} rfc3550_rtcp_type_t;

#define RTCP_FIRST_PAYLOAD_TYPE     200
#define RTCP_LAST_PAYLOAD_TYPE      207

typedef enum
{
    RTCP_SDES_END   = 0,
    RTCP_SDES_CNAME = 1,
    RTCP_SDES_NAME  = 2,
    RTCP_SDES_EMAIL = 3,
    RTCP_SDES_PHONE = 4,
    RTCP_SDES_LOC   = 5,
    RTCP_SDES_TOOL  = 6,
    RTCP_SDES_NOTE  = 7,
    RTCP_SDES_PRIV  = 8,
    RTCP_SDES_MAX   = 9
} rfc3550_rtcp_sdes_type_t;

typedef struct
{
    uint8_t type;                       /* type of item (rfc3550_rtcp_sdes_type_t) */
    uint8_t length;                     /* length of item (in octets) */
    const char *value;                 /* text, not null-terminated */
} rtcp_sdes_item_t;

typedef struct
{
    rtp_rx_packet_handler_t *rx_packet_handler;
    void *user_data;
} rfc3550_payload_handler_t;

/* Per-source state information */
typedef struct
{
    uint32_t ssrc;
    uint16_t max_seq;           /* Highest seq. number seen */
    uint32_t cycles;            /* Shifted count of seq. number cycles */
    uint32_t base_seq;          /* Base seq number */
    uint32_t bad_seq;           /* Last 'bad' seq number + 1 */
    uint32_t probation;         /* Sequ. packets till source is valid */
    uint32_t received;          /* Packets received */
    uint32_t expected_prior;    /* Packet expected at last interval */
    uint32_t received_prior;    /* Packet received at last interval */
    uint32_t transit;           /* Relative trans time for prev pkt */
    uint32_t jitter;            /* Estimated jitter */
    uint64_t last_sender_report_time;
    uint64_t last_sender_report_rx_time;
} rfc3550_source_info_t;

typedef struct
{
    rfc3550_payload_handler_t payload_handler[128];
    rfc3550_payload_handler_t default_payload_handler;

    udp_state_t *rtp_port;
    udp_state_t *rtcp_port;
    /*! The SSRC, stored in network order. */
    uint32_t ssrc;
    /*! The current quantity of CSRCs. */
    int csrc_count;
    /*! The CSRCs, stored in network order. */
    uint32_t csrc[15];
    rfc3550_timestamp_t last_tx_timestamp;
    rfc3550_timestamp_t last_rx_timestamp;
    rfc3550_timestamp_t tx_timestamp_step;
    uint16_t tx_seq_no;
    uint16_t rx_seq_no;
#if defined(ENABLE_SRTP)
    srtp_t srtp;
    rtp_generate_key_cb key_cb;
#endif
    uint32_t tx_packets;
    uint32_t tx_octets;
    
    int avg_rtcp_size;
    
    /* Deal with the common case of a single source locally */
    rfc3550_source_info_t source;
} rfc3550_state_t;

#if defined(__cplusplus)
extern "C"
{
#endif

void rfc3550_set_peer(rfc3550_state_t *s, struct sockaddr_in *them);

void rfc3550_get_peer(rfc3550_state_t *s, struct sockaddr_in *them);

void rfc3550_offered_from_local(rfc3550_state_t *s, int local);

/*! Build an RTP packet */
int rfc3550_build_rtp_packet(rfc3550_state_t *s,
                             const uint8_t msg[],
                             int msg_len,
                             uint8_t buf[],
                             rfc3550_timestamp_t timestamp,
                             int payload_type,
                             int mark);

/*! Build an RTP packet in place. The offet for the start of the data must be far enough into the
    message buffer for the RTP header to fit in front of it. */
int rfc3550_build_rtp_packet_in_place(rfc3550_state_t *s,
                                      uint8_t msg[],
                                      int offset,
                                      rfc3550_timestamp_t timestamp,
                                      int payload_type,
                                      int mark,
                                      int msg_len);

int rfc3550_build_rtcp_packet_start(rfc3550_state_t *s,
                                    uint8_t buf[],
                                    int offset);

int rfc3550_build_rtcp_add_sr_rr_start(rfc3550_state_t *s,
                                       uint8_t buf[],
                                       int offset,
                                       int with_sender);

int rfc3550_build_rtcp_add_sr_rr_chunk(rfc3550_state_t *s,
                                       uint8_t buf[],
                                       int offset,
                                       rfc3550_source_info_t *source);

int rfc3550_build_rtcp_add_sr_rr_end(rfc3550_state_t *s,
                                     uint8_t buf[],
                                     int offset);

int rfc3550_build_rtcp_add_sdes_start(rfc3550_state_t *s,
                                       uint8_t buf[],
                                       int offset);
                                       
int rfc3550_build_rtcp_add_sdes_chunk(rfc3550_state_t *s,
                                      uint8_t buf[],
                                      int offset,
                                      uint32_t ssrc,
                                      int sdes_count,
                                      rtcp_sdes_item_t items[]);

int rfc3550_build_rtcp_add_sdes_end(rfc3550_state_t *s,
                                    uint8_t buf[],
                                    int offset);

int rfc3550_build_rtcp_add_bye(rfc3550_state_t *s,
                               uint8_t buf[],
                               int offset,
                               int sc_count,
                               uint32_t ssrc[],
                               const char *message);

int rfc3550_build_rtcp_add_app(rfc3550_state_t *s,
                               uint8_t buf[],
                               int offset,
                               uint32_t ssrc,
                               const char name[4],
                               const void *data,
                               unsigned int data_len);

int rfc3550_build_rtcp_add_rtpfb(rfc3550_state_t *s,
                                 uint8_t buf[],
                                 int offset,
                                 uint32_t ssrc);

int rfc3550_build_rtcp_add_psfb(rfc3550_state_t *s,
                                uint8_t buf[],
                                int offset,
                                uint32_t ssrc);

int rfc3550_build_rtcp_add_xr(rfc3550_state_t *s,
                              uint8_t buf[],
                              int offset,
                              uint32_t ssrc);

int rfc3550_build_rtcp_packet_end(rfc3550_state_t *s,
                                  uint8_t buf[],
                                  int offset);

int rfc3550_add_csrc(rfc3550_state_t *rtp, uint32_t csrc);

int rfc3550_del_csrc(rfc3550_state_t *rtp, uint32_t csrc);

int rfc3550_find_csrc(rfc3550_state_t *rtp, uint32_t csrc);

uint32_t rfc3550_set_ssrc(rfc3550_state_t *rtp, uint32_t ssrc);

uint32_t rfc3550_get_ssrc(rfc3550_state_t *rtp);

int rfc3550_rx_rtp_packet(rfc3550_state_t *s, const uint8_t buf[], int len);

int rfc3550_rx_rtcp_packet(rfc3550_state_t *s, const uint8_t buf[], int len);

rfc3550_state_t *rfc3550_init(rfc3550_state_t *s, int with_rtcp);

void rfc3550_reset(rfc3550_state_t *s);

int rfc3550_release(rfc3550_state_t *s);

/*!
    \param payload_type Values 0 to 127 set a handler for the specified payload
           type. -1 sets a default handler for types for which no handler has been
           explicitly set
*/
int rfc3550_set_payload_handler(rfc3550_state_t *s,
                                int payload_type,
                                rtp_rx_packet_handler_t rx_packet_handler,
                                void *user_data);

int rfc3550_get_payload_handler(rfc3550_state_t *s,
                                int payload_type,
                                rtp_rx_packet_handler_t **rx_packet_handler,
                                void **user_data);

#if defined(ENABLE_SRTP)

/* Crypto suites */
#define RFC3550_AES_CM_128_HMAC_SHA1_80     1
#define RFC3550_AES_CM_128_HMAC_SHA1_32     2
#define RFC3550_F8_128_HMAC_SHA1_80         3

#define MIKEY_SRTP_EALG_NULL                0
#define MIKEY_SRTP_EALG_AESCM               1
#define MIKEY_SRTP_AALG_NULL                0
#define MIKEY_SRTP_AALG_SHA1HMAC            1

typedef struct rfc3550_policy_s rfc3550_policy_t;
typedef int (*rtp_generate_key_cb)(rfc3550_state_t *rtp, uint32_t ssrc, void *data);

void rfc3550_set_generate_key_cb(rfc3550_state_t *rtp,
                                  rtp_generate_key_cb cb);
int rfc3550_add_policy(rfc3550_state_t *rtp, rfc3550_policy_t *policy);

rfc3550_policy_t *rfc3550_policy_alloc(void);
int rfc3550_policy_set_suite(rfc3550_policy_t *policy, int suite);
int rfc3550_policy_set_master_key(rfc3550_policy_t *policy,
                                  const uint8_t *key,
                                  size_t key_len,
                                  const uint8_t *salt,
                                  size_t salt_len);
int rfc3550_policy_set_encr_alg(rfc3550_policy_t *policy, int ealg);
int rfc3550_policy_set_auth_alg(rfc3550_policy_t *policy, int aalg);
void rfc3550_policy_set_encr_keylen(rfc3550_policy_t *policy, int ekeyl);
void rfc3550_policy_set_auth_keylen(rfc3550_policy_t *policy, int akeyl);
void rfc3550_policy_set_srtp_auth_taglen(rfc3550_policy_t *policy, int autht);
void rfc3550_policy_set_srtp_encr_enable(rfc3550_policy_t *policy, int enable);
void rfc3550_policy_set_srtcp_encr_enable(rfc3550_policy_t *policy, int enable);
void rfc3550_policy_set_srtp_auth_enable(rfc3550_policy_t *policy, int enable);
void rfc3550_policy_set_ssrc(rfc3550_policy_t *policy,
                             rfc3550_state_t *rtp, 
                             uint32_t ssrc,
                             int inbound);
void rfc3550_policy_destroy(rfc3550_policy_t *policy);
int rfc3550_get_random(uint8_t *key, size_t len);

#endif

uint64_t ntptime(void);

struct timeval ntp_to_timeval(uint64_t when);
    
uint64_t timeval_to_ntp(struct timeval *t);

static __inline__ uint32_t ntp_to_reduced_ntp(uint64_t when)
{
    return (uint32_t) ((when >> 16) & (uint64_t) 0xFFFFFFFF);
}
/*- End of function --------------------------------------------------------*/

#if defined(__cplusplus)
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
