/*
 * Vale - a library for media streaming.
 *
 * rfc2733.h - RTP FEC
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc2733.h,v 1.1.1.1 2007/04/03 13:13:37 steveu Exp $
 */

/*! \file */

#if !defined(_VALE_RFC2733_H_)
#define _VALE_RFC2733_H_

#define LOCAL_FAX_MAX_DATAGRAM      400
#define LOCAL_FAX_MAX_FEC_PACKETS   5

#define RFC2733_BUF_MASK            15

enum
{
    RFC2733_ERROR_CORRECTION_NONE,
    RFC2733_ERROR_CORRECTION_FEC,
    RFC2733_ERROR_CORRECTION_REDUNDANCY
};

typedef struct
{
    int buf_len;
    uint8_t buf[LOCAL_FAX_MAX_DATAGRAM];
} rfc2733_fec_tx_buffer_t;

typedef struct
{
    int buf_len;
    uint8_t buf[LOCAL_FAX_MAX_DATAGRAM];
    int fec_len[LOCAL_FAX_MAX_FEC_PACKETS];
    uint8_t fec[LOCAL_FAX_MAX_FEC_PACKETS][LOCAL_FAX_MAX_DATAGRAM];
    int fec_span;
    int fec_entries;
} rfc2733_fec_rx_buffer_t;

typedef int (rfc2733_rx_packet_handler_t)(void *user_data, int seq_no, int rtp_type, const uint8_t msg[], int len);

typedef struct
{
    rfc2733_rx_packet_handler_t *rx_packet_handler;
    void *user_data;

    /*! This option indicates the error correction scheme used in transmitted RFC2733
        packets. */
    int error_correction_scheme;

    /*! This option indicates the number of error correction entries transmitted in
        RFC2733 packets. */
    int error_correction_entries;

    /*! This option indicates the span of the error correction entries in transmitted
        RFC2733 packets (FEC only). */
    int error_correction_span;

    /*! This option indicates the maximum size of a datagram that can be accepted by
        the remote device. */
    int far_max_datagram_size;

    /*! This option indicates the maximum size of a datagram that we are prepared to
        accept. */
    int local_max_datagram_size;

    int verbose;

    int tx_seq_no;
    int rx_seq_no;
    int rx_expected_seq_no;

    rfc2733_fec_tx_buffer_t tx[RFC2733_BUF_MASK + 1];
    rfc2733_fec_rx_buffer_t rx[RFC2733_BUF_MASK + 1];
} rfc2733_state_t;

#if defined(__cplusplus)
extern "C"
{
#endif

int rfc2733_process_rx_packet(void *user_data,
                              uint16_t seq_no,
                              uint32_t timestamp,
                              uint32_t ssrc,
                              int payload_type,
                              int mark,
                              const uint8_t buf[],
                              int len);

int rfc2733_build_packet(rfc2733_state_t *s, uint8_t buf[], const uint8_t msg[], int msg_len);

rfc2733_state_t *rfc2733_init(rfc2733_state_t *s,
                              int ec_scheme,
                              int span,
                              int entries,
                              rfc2733_rx_packet_handler_t rx_packet_handler,
                              void *user_data);

#ifdef __cplusplus
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
