/*
 * Vale - a library for media streaming.
 *
 * queue.h - Packet queuing and buffer management.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: queue.h,v 1.3 2007/04/07 03:37:25 steveu Exp $
 */


/*! \file */

#if !defined(_VALE_QUEUE_H_)
#define _VALE_QUEUE_H_

typedef struct frame_queue_frame_s
{
    /*! Pointer to the next earlier frame */
    struct frame_queue_frame_s *earlier;
    /*! Pointer to the next later frame */
    struct frame_queue_frame_s *later;
    /*! The type of frame */
    int payload_type;
    /*! The sequence number of the frame */
    uint16_t seq_no;
    /*! The timestamp assigned by the sending end */
    rfc3550_timestamp_t sender_stamp;
    /*! The timespan covered by the data in this frame */
    rfc3550_timestamp_t sender_len;
    /*! The timestamp assigned by the receiving end */
    rfc3550_timestamp_t receiver_stamp;
    int len;
    /*! The actual frame data */
    uint8_t data[];
} frame_queue_frame_t;

typedef struct
{
    /*! The total frames input to the buffer, to date. */
    int frames_in;
    /*! The total frames output from the buffer, to date. */
    int frames_out;
    /*! The number of frames received out of sequence. */
    int frames_oos;
    /*! The number of frames repeated. */
    int frames_repeats;
    /*! The number of frames which were discarded, due to late arrival. */
    int frames_late;
    /*! The number of frames which were never received. */
    int frames_missing;
    /*! The number of frames trimmed from the stream, due to buffer shrinkage. */
    int frames_trimmed;
} frame_queue_stats_t;

typedef struct
{
    /*! The queued frame list */
    frame_queue_frame_t *first_frame;
    frame_queue_frame_t *last_frame;
    /*! The free frame pool */
    frame_queue_frame_t *free_frames;
    
    int data_size;

    frame_queue_stats_t stats;
} frame_queue_state_t;

#if defined(__cplusplus)
extern "C"
{
#endif

frame_queue_frame_t *frame_queue_get_free_frame(frame_queue_state_t *s);

void frame_queue_put_free_frame(frame_queue_state_t *s, frame_queue_frame_t *frame);

void frame_queue_fill_frame(frame_queue_frame_t *frame,
                            uint8_t data[],
                            int len,
                            int payload_type,
                            int seq_no,
                            rfc3550_timestamp_t sender_len,
                            rfc3550_timestamp_t sender_stamp,
                            rfc3550_timestamp_t receiver_stamp);

int frame_queue_put_frame(frame_queue_state_t *s, frame_queue_frame_t *frame);

frame_queue_frame_t *frame_queue_get_frame(frame_queue_state_t *s, rfc3550_timestamp_t sender_stamp);

int frame_queue_put_rx_packet(void *user_data,
                              const uint8_t buf[],
                              int len,
                              uint16_t seq_no,
                              uint32_t timestamp,
                              uint32_t ssrc,
                              int payload_type,
                              int mark);

int frame_queue_free_frames(frame_queue_state_t *s);

int frame_queue_queued_frames(frame_queue_state_t *s);

void frame_queue_restart(frame_queue_state_t *s, int data_size);

frame_queue_state_t *frame_queue_new(int data_size);

void frame_queue_free(frame_queue_state_t *s);

void frame_queue_stats(frame_queue_state_t *s, frame_queue_stats_t *stats);

void frame_queue_dump(frame_queue_state_t *s);

#if defined(__cplusplus)
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
