/*
 * Vale - a library for media streaming.
 *
 * rfc4733.c - RTP payload for DTMF digits, telephony tones and telephony signals
 *             (An update of RFC2833) (includes event definitions from RFC4734)
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc4733.c,v 1.3 2007/09/22 12:17:31 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <inttypes.h>
#if defined(HAVE_TGMATH_H)
#include <tgmath.h>
#endif
#if defined(HAVE_MATH_H)
#include <math.h>
#endif
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#ifdef ENABLE_SRTP
#include <srtp.h>
#endif

#include "vale/unaligned.h"
#include "vale/udp.h"
#include "vale/rfc3550.h"
#include "vale/rfc4733.h"

#define FALSE 0
#define TRUE (!FALSE)

/*
    0                   1                   2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     event     |E|R| volume    |          duration             |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

               Figure 1: Payload Format for Named Events


    0                   1                   2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |V=2|P|X|  CC   |M|     PT      |       sequence number         |
   | 2 |0|0|   0   |0|     96      |              28               |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                           timestamp                           |
   |                             11200                             |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |           synchronization source (SSRC) identifier            |
   |                            0x5234a8                           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |F|   block PT  |     timestamp offset      |   block length    |
   |1|     97      |            11200          |         4         |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |F|   block PT  |     timestamp offset      |   block length    |
   |1|     97      |   11200 - 6400 = 4800     |         4         |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |F|   Block PT  |
   |0|     97      |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     digit     |E R| volume    |          duration             |
   |       9       |1 0|     7     |             1600              |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     digit     |E R| volume    |          duration             |
   |       1       |1 0|    10     |             2000              |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     digit     |E R| volume    |          duration             |
   |       1       |0 0|    20     |              400              |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

          Figure 2: Example RTP packet after dialing "911"


     0                   1                   2                   3
     0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |    modulation   |T|  volume   |          duration             |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |R R R R|       frequency       |R R R R|       frequency       |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |R R R R|       frequency       |R R R R|       frequency       |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    ......

    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |R R R R|       frequency       |R R R R|      frequency        |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

                 Figure 3: Payload format for tones


     0                   1                   2                   3
     0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    | V |P|X|  CC   |M|     PT      |       sequence number         |
    | 2 |0|0|   0   |0|     96      |              31               |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |                           timestamp                           |
    |                             48000                             |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |           synchronization source (SSRC) identifier            |
    |                            0x5234a8                           |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |F|   block PT  |     timestamp offset      |   block length    |
    |1|     98      |            16383          |         4         |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |F|   block PT  |     timestamp offset      |   block length    |
    |1|     97      |            16383          |         8         |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |F|   Block PT  |
    |0|     97      |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |  event=ring   |0|0| volume=0  |     duration=28383            |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    | modulation=0    |0| volume=63 |     duration=16383            |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |0 0 0 0|     frequency=0       |0 0 0 0|    frequency=0        |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    | modulation=0    |0| volume=5  |     duration=12000            |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |0 0 0 0|     frequency=440     |0 0 0 0|    frequency=480      |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

       Figure 4: Combining tones and events in a single RTP packet
*/

const char *rfc4733_event_to_str(int event)
{
    switch (event)
    {
    case RFC4733_EVENT_DTMF_0:
        return "DTMF-0";
    case RFC4733_EVENT_DTMF_1:
        return "DTMF-1";
    case RFC4733_EVENT_DTMF_2:
        return "DTMF-2";
    case RFC4733_EVENT_DTMF_3:
        return "DTMF-3";
    case RFC4733_EVENT_DTMF_4:
        return "DTMF-4";
    case RFC4733_EVENT_DTMF_5:
        return "DTMF-5";
    case RFC4733_EVENT_DTMF_6:
        return "DTMF-6";
    case RFC4733_EVENT_DTMF_7:
        return "DTMF-7";
    case RFC4733_EVENT_DTMF_8:
        return "DTMF-8";
    case RFC4733_EVENT_DTMF_9:
        return "DTMF-9";
    case RFC4733_EVENT_DTMF_STAR:
        return "DTMF-Star";
    case RFC4733_EVENT_DTMF_HASH:
        return "DTMF-Hash";
    case RFC4733_EVENT_DTMF_A:
        return "DTMF-A";
    case RFC4733_EVENT_DTMF_B:
        return "DTMF-B";
    case RFC4733_EVENT_DTMF_C:
        return "DTMF-C";
    case RFC4733_EVENT_DTMF_D:
        return "DTMF-D";
    case RFC4733_EVENT_DTMF_HOOK_FLASH:
        return "Hook flash";
    case RFC4733_EVENT_CED:
        return "CED";
    case RFC4733_EVENT_CNG:
        return "CNG";
    case RFC4733_EVENT_V21_CH1_0:
        return "V.21 Ch1 0";
    case RFC4733_EVENT_V21_CH1_1:
        return "V.21 Ch1 1";
    case RFC4733_EVENT_V21_CH2_0:
        return "V.21 Ch2 0";
    case RFC4733_EVENT_V21_CH2_1:
        return "V.21 Ch2 1";
    }
    return "???";
}

static int rfc4733_process_rx_event_packet(void *user_data,
                                           const uint8_t msg[],
                                           int len,
                                           uint16_t seq_no,
                                           uint32_t timestamp,
                                           uint32_t ssrc,
                                           int payload_type,
                                           int mark)
{
    uint32_t event;
    uint32_t volume;
    uint32_t event_end;
    uint32_t duration;
    uint32_t tag;
    char resp = 0;
    rfc4733_state_t *s;
    /* X means hook flash */
    static const char events[17] = "0123456789*#ABCDX";

    s = (rfc4733_state_t *) user_data;
    if (len < sizeof(uint32_t))
        return -1;
    tag = ntohl(get_unaligned_uint32(msg));
    event = tag >> 24;
    event_end = (tag >> 23) & 1;
    volume = (tag >> 16) & 0x3F;
    duration = tag & 0xFFFF;
    resp = (event < 17)  ?  events[event]  :  -1;
    if (!event_end)
        s->resp = resp;
    if (s->handler)
        s->handler(s->user_data, volume, duration, event_end, event, NULL);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int rfc4733_process_rx_tone_packet(void *user_data,
                                          const uint8_t msg[],
                                          int len,
                                          uint16_t seq_no,
                                          uint32_t timestamp,
                                          uint32_t ssrc,
                                          int payload_type,
                                          int mark)
{
    uint32_t modulation_freq;
    uint32_t mult3;
    uint32_t volume;
    uint32_t duration;
    uint32_t tag;
    rfc4733_state_t *s;
    int frequencies[16];
    int offset;
    int i;

    s = (rfc4733_state_t *) user_data;
    if (len < sizeof(uint32_t))
        return -1;
    tag = ntohl(get_unaligned_uint32(msg));
    modulation_freq = tag >> 23;
    mult3 = (tag >> 22) & 1;
    volume = (tag >> 16) & 0x3F;
    duration = tag & 0xFFFF;
    for (offset = sizeof(uint32_t), i = 0;  offset < len  &&  i < 16;  offset += sizeof(uint16_t), i++)
        frequencies[i] = ntohs(get_unaligned_uint16(msg + offset));
    if (s->handler)
        s->handler(s->user_data, volume, duration, modulation_freq, i, frequencies);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int cisco_dtmf_process_rx_packet(void *user_data,
                                        const uint8_t msg[],
                                        int len,
                                        uint16_t seq_no,
                                        uint32_t timestamp,
                                        uint32_t ssrc,
                                        int payload_type,
                                        int mark)
{
    int event;
    char resp = 0;
    rfc4733_state_t *s;
    static const char events[17] = "0123456789*#ABCDX";

    s = (rfc4733_state_t *) user_data;
    event = ntohl(get_unaligned_uint32(msg)) & 0x001F;
    resp = (event < 17)  ?  events[event]  :  -1;
    if (s->handler)
    {
        s->handler(s->user_data, 42, 800, FALSE, event, NULL);
        s->handler(s->user_data, 42, 0, TRUE, event, NULL);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc4733_event_send(rfc4733_state_t *s, uint8_t buf[], int event, int volume, int duration, int end)
{
    int len;
    uint8_t msg[sizeof(uint32_t)];
    
    if (s->event_payload_type < 0)
        return -1;
    put_unaligned_uint32(msg, htonl((event << 24) | (end << 23) | (volume << 16) | duration));

    len = rfc3550_build_rtp_packet(s->rfc3550_state,
                                   msg,
                                   sizeof(uint32_t),
                                   buf,
                                   0,
                                   s->event_payload_type,
                                   0);
    return len;
}
/*- End of function --------------------------------------------------------*/

int rfc4733_tone_send(rfc4733_state_t *s, uint8_t buf[], int modulation, int volume, int duration, int freqs, int freq[])
{
    int i;
    int len;
    uint8_t msg[sizeof(uint32_t) + 16*sizeof(uint16_t)];
    int mult3;
    int offset;
    
    if (s->tone_payload_type < 0)
        return -1;
    mult3 = 0;
    put_unaligned_uint32(msg, htonl((modulation << 22) | (mult3 << 22) | (volume << 16) | duration));
    for (offset = sizeof(uint32_t), i = 0;  i < freqs;  offset += sizeof(uint16_t), i++)
        put_unaligned_uint16(msg + offset, ntohs(freq[i]));
    if (freqs & 1)
    {
        put_unaligned_uint16(msg + offset, 0);
        offset += sizeof(uint16_t);
    }
    len = rfc3550_build_rtp_packet(s->rfc3550_state,
                                   msg,
                                   offset,
                                   buf,
                                   0,
                                   s->tone_payload_type,
                                   0);
    return len;
}
/*- End of function --------------------------------------------------------*/

int rfc4733_cisco_dtmf_send(rfc4733_state_t *s, uint8_t buf[], int event)
{
    int len;
    uint8_t msg[sizeof(uint32_t)];
    
    if (s->cisco_dtmf_payload_type < 0)
        return -1;
    put_unaligned_uint32(msg, htonl(event));

    len = rfc3550_build_rtp_packet(s->rfc3550_state,
                                   msg,
                                   sizeof(uint32_t),
                                   buf,
                                   0,
                                   s->cisco_dtmf_payload_type,
                                   0);
    return len;
}
/*- End of function --------------------------------------------------------*/

int rfc4733_set_cisco_dtmf_type(rfc4733_state_t *s, int cisco_dtmf_pt)
{
    if (cisco_dtmf_pt >= 0)
        rfc3550_set_payload_handler(s->rfc3550_state, cisco_dtmf_pt, cisco_dtmf_process_rx_packet, (void *) s);
    s->cisco_dtmf_payload_type = cisco_dtmf_pt;
    return 0;
}
/*- End of function --------------------------------------------------------*/

rfc4733_state_t *rfc4733_init(rfc4733_state_t *s,
                              rfc3550_state_t *t,
                              int event_pt,
                              int tone_pt,
                              rfc4733_event_handler_t *handler,
                              void *user_data)
{
    if (s == NULL)
    {
        if ((s = (rfc4733_state_t *) malloc(sizeof(*s))) == NULL)
            return NULL;
    }
    memset(s, 0, sizeof(*s));
    s->rfc3550_state = t;
    /* Register the payload types */
    s->event_payload_type = event_pt;
    if (event_pt >= 0)
        rfc3550_set_payload_handler(t, event_pt, rfc4733_process_rx_event_packet, (void *) s);
    s->tone_payload_type = tone_pt;
    if (tone_pt >= 0)
        rfc3550_set_payload_handler(t, tone_pt, rfc4733_process_rx_tone_packet, (void *) s);
    s->cisco_dtmf_payload_type = -1;
    s->handler = handler;
    s->user_data = user_data;
    return s;
}
/*- End of function --------------------------------------------------------*/

int rfc4733_release(rfc4733_state_t *s)
{
    if (s->event_payload_type >= 0)
        rfc3550_set_payload_handler(s->rfc3550_state, s->event_payload_type, NULL, NULL);
    if (s->tone_payload_type >= 0)
        rfc3550_set_payload_handler(s->rfc3550_state, s->tone_payload_type, NULL, NULL);
    if (s->cisco_dtmf_payload_type >= 0)
        rfc3550_set_payload_handler(s->rfc3550_state, s->cisco_dtmf_payload_type, NULL, NULL);
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
