#include <math.h>
#include <FL/Fl.H>
//#include <FL/Fl_Audio_Meter.H>
#include "Fl_Audio_Meter.H"
#include <FL/fl_draw.H>

//
// Fl_Audio_Meter is an audio meter widget based off Fl_Widget
//

static float iec_scale(float db)
{
    float def = 0.0f;   /* Meter deflection %age */

    if (db < -70.0f)
        def = 0.0f;
    else if (db < -60.0f)
        def = (db + 70.0f)*0.25f;
    else if (db < -50.0f)
        def = (db + 60.0f)*0.5f + 5.0f;
    else if (db < -40.0f)
        def = (db + 50.0f)*0.75f + 7.5f;
    else if (db < -30.0f)
        def = (db + 40.0f)*1.5f + 15.0f;
    else if (db < -20.0f)
        def = (db + 30.0f)*2.0f + 30.0f;
    else if (db < 0.0f)
        def = (db + 20.0f)*2.5f + 50.0f;
    else
        def = 100.0f;
    return def;
}

void Fl_Audio_Meter::sample_rate(float v)
{
    sample_rate_ = v;
    sample_count =
    samples_per_update = (int) (sample_rate_/25.0);
    short_term_damping = 400.0/v;
    peak_damping = 1.0 - 2.0/v;
}

void Fl_Audio_Meter::sample(float val)
{
    /* Do a short term energy follower */
    val *= val;
    current_value_ += (val - current_value_)*short_term_damping;
    /* Always follow the short term signal upwards, and
       decay gentle when the short term signal is below
       our peak follower */
    if (peak_value_ < current_value_)
        peak_value_ = current_value_;
    else
        peak_value_ *= peak_damping;
    if (--sample_count <= 0)
    {
        redraw();
        sample_count = samples_per_update;
    }
}

//
// 'Fl_Audio_Meter::draw()' - Draw the check button.
//

void Fl_Audio_Meter::draw()
{
    float ii;
    float pp;
    int	level_i;
    int	level_p;
    int	bx;
    int by;
    int bw;
    int bh;
    int	tx;
    int ty;
    int	tw;
    int th;
    int pos;
    int half_width;
    int start;
    int end;
    int i;
    int j;
    static const float db_steps[] = {0.025, 0.075, 0.15, 0.3, 0.5, 0.75, 1.0, -1.0};

    // Get the box borders...
    bx = Fl::box_dx(box());
    by = Fl::box_dy(box());
    bw = Fl::box_dw(box());
    bh = Fl::box_dh(box());

    if (current_value_ > 0.0f)
        ii = iec_scale(10.0f*log10f(current_value_*gain_))*0.01;
    else
        ii = 0.01;
    if (peak_value_ > 0.0f)
        pp = iec_scale(10.0f*log10f(peak_value_*gain_))*0.01;
    else
        pp = 0.01;
    
    
    if ((type() & 1))
    {
        /* Horizontal */
        tx = x() + bx;
        tw = w() - bw;
        half_width = (h() - bh) >> 2;

        level_i = (int) (ii*tw + 0.5);
        level_p = (int) (pp*tw + 0.5);
    
        if (level_i)
        {
            level_i += bx;
            fl_clip(x(), y(), level_i, h());
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  color2()  :  fl_inactive(color2()));
            fl_pop_clip();
        }
        if (level_p > level_i)
        {
            level_p += bx;
            fl_clip(x() + level_i, y(), level_p - level_i, h());
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  peak_color()  :  fl_inactive(peak_color()));
            fl_pop_clip();
            fl_clip(x() + level_p, y(), w() - level_p, h());
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  color()  :  fl_inactive(color()));
            fl_pop_clip();
        }
        else
        {
            fl_clip(x() + level_i, y(), w() + bx - level_i, h());
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  color()  :  fl_inactive(color()));
            fl_pop_clip();
        }
        if (display_ticks_)
        {
            /* Now display all the dB ticks */
            start = y() + by;
            end = y() + by + h() - bh - 1;
            /* Do the 10dB step markers */
            for (i = 0;  i < 7;  i++)
            {
                pos = tx + (int) (tw*db_steps[i] + 0.5);
                fl_line(pos, start, pos, end);
            }
            /* Fill in some interventing markers */
            for (i = 1;  i <= 9;  i++)
            {
                pos = tx + (int) (tw*(0.30 + i*0.02) + 0.5);
                fl_line(pos, start + half_width, pos, end - half_width);
            }
            for (j = 0;  j < 20;  j += 10)
            {
                for (i = 1;  i <= 9;  i++)
                {
                    pos = tx + (int) (tw*(0.5 + (i + j)*0.025) + 0.5);
                    fl_line(pos, start + half_width, pos, end - half_width);
                }
            }
        }
    }
    else
    {
        /* Vertical */
        ty = y() + h();
        th = h() - bh;
        half_width = (w() - bw) >> 2;

        level_i = (int) (ii*th + 0.5);
        level_p = (int) (pp*th + 0.5);
    
        if (level_i)
        {
            level_i += by;
            fl_clip(x(), ty - level_i, w(), level_i);
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  color2()  :  fl_inactive(color2()));
            fl_pop_clip();
        }
        if (level_p > level_i)
        {
            level_p += by;
            fl_clip(x(), ty - level_p, w(), level_p - level_i);
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  peak_color()  :  fl_inactive(peak_color()));
            fl_pop_clip();
            fl_clip(x(), y(), w(), h() - level_p);
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  color()  :  fl_inactive(color()));
            fl_pop_clip();
        }
        else
        {
            fl_clip(x(), y(), w(), h() - level_i);
            draw_box(box(), x(), y(), w(), h(), active_r()  ?  color()  :  fl_inactive(color()));
            fl_pop_clip();
        }
        if (display_ticks_)
        {
            /* Now display all the dB ticks */
            start = x() + bx;
            end = x() + bx + w() - bw - 1;
            /* Do the 10dB step markers */
            for (i = 0;  i < 7;  i++)
            {
                pos = ty - by - (int) (th*db_steps[i] + 0.5);
                fl_line(start, pos, end, pos);
            }
            /* Fill in some interventing markers */
            for (i = 1;  i <= 9;  i++)
            {
                pos = ty - by - (int) (th*(0.30 + i*0.02) + 0.5);
                fl_line(start + half_width, pos, end - half_width, pos);
            }
            for (j = 0;  j < 20;  j += 10)
            {
                for (i = 1;  i <= 9;  i++)
                {
                    pos = ty - by - (int) (th*(0.5 + (i + j)*0.025) + 0.5);
                    fl_line(start + half_width, pos, end - half_width, pos);
                }
            }
        }
    }
}

//
// 'Fl_Audio_Meter::Fl_Audio_Meter()' - Construct a Fl_Audio_Meter widget.
//

Fl_Audio_Meter::Fl_Audio_Meter(int X, int Y, int W, int H, const char *l)
: Fl_Widget(X, Y, W, H, l)
{
    align(FL_ALIGN_INSIDE);
    box(FL_DOWN_BOX);
    color(FL_BACKGROUND2_COLOR, FL_GREEN);
    sample_rate(8000.0);
    gain(1.0);
    current_value(0.0f);
    peak_value(0.0f);
    color(FL_BLACK, FL_RED);
    peak_color(FL_YELLOW);
    display_ticks(true);
}
